/**
* \file: osgCamtoGstbuf.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Abstract base class for camtogst for OSG
*
* \component: osgCamtoGstbuf
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>, Anil V<anil.valmiki@in.bosch.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/
#include <osg/RenderInfo>
#include <osg/Camera>
#include <osgViewer/api/Wayland/GraphicsWindowWayland>
#include <EGL/eglext.h>
#include <osg/FrameBufferObject>
#include <osg/Texture2D>

#include <sstream>
#include <cassert>
#include <ctime>
#include <memory>
#include <queue>

#include "osgcamtogstbuf.h"

static const char GST_BUFFER_QUARK[] = "Release-textureobject-helper\0";

namespace osgCamtoGstbuf {

class Texture_obj : public osg::Texture2D
{
public:
    Texture_obj(uint context_ID):osg::Texture2D(),txo(), context_id(context_ID){}
    osg::ref_ptr<osg::Texture::TextureObject> txo;
    uint tex_id;
    uint tex_width;
    uint tex_height;
    uint context_id;
    void clearTextureObject(uint context_id);
};

#if GST_CHECK_VERSION(1,0,0)

void OsgCamtoGstbuf::gst_buffer_final (struct UserPtrData *data)
{
    Texture_obj *_txo = (osgCamtoGstbuf::Texture_obj*)data->tex;
    OsgCamtoGstbuf *_camera = (osgCamtoGstbuf::OsgCamtoGstbuf*)data->osg_cam;

    _txo->clearTextureObject(_txo->context_id);
    _camera->releaseTex(_txo->tex_id);
    _txo->txo = NULL;
    _txo->unref();
}
#else
/**
 * @brief Helper class which is used as qdata on a GstBuffer
 * The qdata will be destroyed when the GstBuffer is released and we use this
 * to mark the Texture object that is sharing its data with the
 * GstBuffer as usable again by putting it back into the pool.
 *
 * Additionally it will also free the GstMeta(releaseTXO) that is attached to the GstBuffer
 * to mark the buffer as physically backed.
 */
class PoolData
{
public:
    PoolData(Texture_obj *txo, OsgCamtoGstbuf *camera)
        : _refCount(1), _txo(txo), _camera(camera){}
    ~PoolData()
    {
        _txo->clearTextureObject(_txo->context_id);
        _camera->releaseTex(_txo->tex_id);
        _txo->txo = NULL;
        _txo->unref();
        _camera->texobj_queue.pop();
    }

    PoolData *ref()
    {
        g_atomic_int_inc(&_refCount);
        return this;
    }

    void unref()
    {
        if (g_atomic_int_dec_and_test(&_refCount))
        {
            delete this;
        }
    }

    /// copy function for GBoxed types
    static void *copy(void *data)
    {
        return reinterpret_cast<PoolData*>(data)->ref();
    }

    /// free function for GBoxed types
    static void free(void *data)
    {
        reinterpret_cast<PoolData*>(data)->unref();
    }

    volatile gint _refCount;
    Texture_obj *_txo;
    OsgCamtoGstbuf *_camera;
};

G_DEFINE_BOXED_TYPE(PoolData, pool_data, PoolData::copy, PoolData::free)
#endif

/**
 * @brief Common base class for the used draw callbacks
 * The class supports chaining to another call-back.
 */
class camtogstDrawCallback : public osg::Camera::DrawCallback
{
public:
    mutable Texture_obj *obj = NULL;
    camtogstDrawCallback(OsgCamtoGstbuf *camtogst, osg::Camera::DrawCallback *chainup)
        : osg::Camera::DrawCallback()
        , _camtogst(camtogst)
        , _chainup(chainup)
    {
    }

    virtual osg::Camera::DrawCallback *getChainup() const
    {
        return _chainup.get();
    }

protected:
    osgCamtoGstbuf::OsgCamtoGstbuf *_camtogst;
    osg::ref_ptr<osg::Camera::DrawCallback> _chainup;
};

/**
 * @brief Pre-draw camera callback
 *
 * Since it is more performant to replace the FBO we cannot use the high-level
 * RTT functionality. Binding the FBO as a state attribute happens too late, the
 * glClear() of the camera will end up on the default framebuffer instead of the
 * FBO.
 *
 * This will request the next free FBO from the camtogst, bind it and remember it
 * for the post-draw callback.
 */
class camtogstRTTPreDrawCallback : public camtogstDrawCallback
{
public:
    camtogstRTTPreDrawCallback(OsgCamtoGstbuf *camtogst, osg::Camera::DrawCallback *other = 0)
        : camtogstDrawCallback(camtogst, other), _fbo(0)
    {
    }

    virtual ~camtogstRTTPreDrawCallback();


    virtual void operator() (osg::RenderInfo& renderInfo) const
    {
        /* enable below lines for more info on errors*/
        //osg::State &state = *renderInfo.getState();
        //state.setCheckForGLErrors(osg::State::ONCE_PER_ATTRIBUTE);

        if(_camtogst)
        {
            const OsgCamtoGstbuf::FboTraits &traits = _camtogst->getFramebufferTraits();

            if(!_fbo)
            {
                _fbo = new osg::FrameBufferObject;
            }

            obj = new Texture_obj(renderInfo.getContextID());
            obj->ref();

            //Textureinfo
            obj->tex_id = _camtogst->nextTex(-1);
            if(obj->tex_id && _camtogst)
            {
                obj->tex_width = _camtogst->getTexWidth();
                obj->tex_height = _camtogst->getTexHeight();

                obj->setTextureSize(obj->tex_width, obj->tex_height);
                obj->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
                obj->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
                obj->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_EDGE);
                obj->setWrap(osg::Texture::WRAP_T, osg::Texture::CLAMP_TO_EDGE);

                obj->txo = new osg::Texture::TextureObject(obj, obj->tex_id, GL_TEXTURE_2D);
                obj->setTextureObject(obj->context_id, obj->txo);
                obj->txo->setAllocated(true);

                _camtogst->rel_obj.obj = obj;
                _camtogst->texobj_queue.push(_camtogst->rel_obj);

                _fbo->setAttachment(osg::Camera::COLOR_BUFFER,
                                    osg::FrameBufferAttachment(obj, 0, traits.samples));

                if(!_rb.valid())
                {
                    if (traits.stencil > 0)
                    {
                        _rb = new osg::RenderBuffer(obj->tex_width, obj->tex_height, GL_DEPTH24_STENCIL8_EXT, traits.samples);
                    }
                    else
                    {
                        if (traits.depth > 0)
                        {
                            _rb = new osg::RenderBuffer(obj->tex_width, obj->tex_height, GL_DEPTH_COMPONENT24, traits.samples);
                        }
                    }
                }

                if (_rb.valid())
                {
                    _fbo->setAttachment(osg::Camera::DEPTH_BUFFER, osg::FrameBufferAttachment(_rb.get()));
                }

                if (traits.stencil > 0)
                {
                    _fbo->setAttachment(osg::Camera::STENCIL_BUFFER, osg::FrameBufferAttachment(_rb.get()));
                }

                const_cast<camtogstRTTPreDrawCallback*>(this)->_tx_obj = obj;
                _fbo->apply(*renderInfo.getState());

                if (_chainup.valid())
                {
                    _chainup->operator()(renderInfo);
                }
            }
            else
            {
                std::cout<<" In valid texture ID or camtogst object destroyed \n"<<std::endl;
                std::cout<< " obj->tex_id "<<obj->tex_id<<std::endl;
                obj->unref();
                pthread_exit(0);
            }
        }
    }

    Texture_obj *getTxobj() const
    {
        return _tx_obj;
    }

private:
    mutable osg::ref_ptr<osg::FrameBufferObject> _fbo;
    mutable osg::ref_ptr<osg::RenderBuffer> _rb;
    Texture_obj *_tx_obj;
};

camtogstRTTPreDrawCallback::~camtogstRTTPreDrawCallback()
{
    _chainup = NULL;
    _camtogst = NULL;
    _fbo = NULL;
    _rb = NULL;
}


/**
 * @brief Post-draw camera callback
 *
 * Manually unbind the FBO bound in the pre-draw callback and tell the camtogst
 * that it is ready to be encoded.
 */
class camtogstRTTFinalDrawCallback : public camtogstDrawCallback
{
public:
    int status;

    struct sync_parameters
    {
        sync_parameters() : curr_txo(0), curr_fence(0){}
        mutable Texture_obj *curr_txo;
        mutable EGLSyncKHR curr_fence;
    }sync_params;

    mutable std::queue<sync_parameters> myqueue;

    static void* wait_client(void *);
    mutable EGLDisplay dpy;
    mutable EGLSyncKHR fence;
    bool release_thread;

    camtogstRTTFinalDrawCallback(osgCamtoGstbuf::OsgCamtoGstbuf *camtogst,
                               camtogstRTTPreDrawCallback *callback,
                               osg::Camera::DrawCallback *other = 0)
        :camtogstDrawCallback(camtogst, other),
         release_thread(false),
         _callback(callback)
        {

        pthread_mutex_init(&_camtogst->sync_lock, NULL);
        pthread_cond_init(&_camtogst->sync_cond, NULL);
        status = pthread_create(&camtogst->thread, NULL, camtogstRTTFinalDrawCallback::wait_client, (void*)this);
        if(0 != status){
            fprintf(stderr, "Failed to create thread: %d \n", status);
        }
    }

    virtual ~camtogstRTTFinalDrawCallback();

    virtual void operator() (osg::RenderInfo& renderInfo) const
    {
        if (_chainup.valid())
        {
            _chainup->operator()(renderInfo);
        }

        dpy = eglGetCurrentDisplay();
        if(EGL_NO_DISPLAY == dpy){
            printf("Failed to obtain current display: %x \n", eglGetError());
        }

        // Unbind framebuffer
        osg::FBOExtensions* ext = osg::FBOExtensions::instance(renderInfo.getContextID(),true);
        ext->glBindFramebuffer(osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER, 0);

        PFNEGLCREATESYNCKHRPROC  eglCreateSyncKHR = (PFNEGLCREATESYNCKHRPROC) eglGetProcAddress("eglCreateSyncKHR");
        fence = eglCreateSyncKHR(dpy, EGL_SYNC_FENCE_KHR, NULL);
        if (EGL_NO_SYNC_KHR == fence){
            printf("EGL FENCE: Failed to create fence: %x\n", eglGetError());
        }

        pthread_mutex_lock(&_camtogst->sync_lock);
        sync_params.curr_txo = _callback->getTxobj();
        sync_params.curr_fence = fence;
        myqueue.push(sync_params);
        pthread_cond_signal(&_camtogst->sync_cond);
        pthread_mutex_unlock(&_camtogst->sync_lock);
    }
private:
    camtogstRTTPreDrawCallback *_callback;
};

camtogstRTTFinalDrawCallback::~camtogstRTTFinalDrawCallback()
{
    pthread_mutex_lock(&_camtogst->sync_lock);
    release_thread = true;
    while(!(myqueue.empty()))
        myqueue.pop();
    pthread_cond_signal(&_camtogst->sync_cond);
    pthread_mutex_unlock(&_camtogst->sync_lock);
    pthread_join(_camtogst->thread, NULL);
    _camtogst = NULL;
}

void* camtogstRTTFinalDrawCallback::wait_client(void *data)
{
    pthread_setname_np(pthread_self(), "wait_client");
    camtogstRTTFinalDrawCallback *handle = static_cast<camtogstRTTFinalDrawCallback *>(data);
    Texture_obj *curr_txo;
    EGLint value = 0;
    EGLint result = EGL_FALSE;
    EGLSyncKHR curr_fence;
    sync_parameters temp;

    while (!handle->release_thread)
    {
        curr_txo = NULL;
        if(handle->_camtogst != NULL)
        {
            pthread_mutex_lock(&(handle->_camtogst->sync_lock));
            pthread_cond_wait(&(handle->_camtogst->sync_cond), &(handle->_camtogst->sync_lock));
            if(!(handle->myqueue.empty()))
            {
                temp = handle->myqueue.front();
                curr_txo = temp.curr_txo;
                curr_fence = temp.curr_fence;
                handle->myqueue.pop();
            }
            pthread_mutex_unlock(&(handle->_camtogst->sync_lock));

            if(handle->release_thread)
            {
                pthread_exit(0);
            }

            if(curr_txo)
            {
                PFNEGLCLIENTWAITSYNCKHRPROC  eglClientWaitSyncKHR = (PFNEGLCLIENTWAITSYNCKHRPROC) eglGetProcAddress("eglClientWaitSyncKHR");
                result = eglClientWaitSyncKHR(handle->dpy, curr_fence, EGL_SYNC_FLUSH_COMMANDS_BIT_KHR, EGL_FOREVER_KHR);
                if (EGL_FALSE == result) {
                    printf("EGL FENCE: Error in waiting for the fence: %x\n", eglGetError());
                }
                PFNEGLGETSYNCATTRIBKHRPROC  eglGetSyncAttribKHR = (PFNEGLGETSYNCATTRIBKHRPROC) eglGetProcAddress("eglGetSyncAttribKHR");
                result = eglGetSyncAttribKHR(handle->dpy, curr_fence, EGL_SYNC_STATUS_KHR, &value);
                if (EGL_FALSE == result) {
                    printf("EGL FENCE: Error in GetSyncAttrib for the fence: %x\n", eglGetError());
                }
                if(handle->release_thread)
                {
                    pthread_exit(0);
                }

                /* pushing to application */
                if(EGL_SIGNALED_KHR == value) {
                    handle->_camtogst->push(curr_txo);
                }

                if(EGL_NO_SYNC_KHR != curr_fence)
                {
                    PFNEGLDESTROYSYNCKHRPROC  eglDestroySyncKHR = (PFNEGLDESTROYSYNCKHRPROC) eglGetProcAddress("eglDestroySyncKHR");
                    eglDestroySyncKHR(handle->dpy, curr_fence);
                }
            }
        }
        else
        {
            pthread_exit(0);
        }
    }
    pthread_exit(0);
}

/**
 * @brief PIMPL class for platform camtogst
 */
class OsgCamtoGstbuf::OsgCamtoGstbufPrivate
{
public:
    OsgCamtoGstbufPrivate(OsgCamtoGstbuf *q)
        : _q(q)
        , _camera()
        , _width(800)
        , _height(480)
        , _fboTraits()
		, qhandle(NULL)
    {}

    ~OsgCamtoGstbufPrivate()
     {
     }

    OsgCamtoGstbuf *_q;
    osg::ref_ptr<osg::Camera> _camera;
    uint _width;
    uint _height;
    OsgCamtoGstbuf::FboTraits _fboTraits;
    camtogstRTTFinalDrawCallback *qhandle;

};

/**
 * @brief set the width and height of the buffer
 * @param width new width,height new height
 */
void OsgCamtoGstbuf::setBuffersize(uint width, uint height)
{
    if((0 != width) && (0 != height))
    {
        setTexSize(width,height);
        uint _height = 0;
        uint _width = 0;
        getTexSize(&_width,&_height);
        _p->_width = _width;
        _p->_height = _height;

        if (_p->_camera.valid())
        {
            _p->_camera->setViewport(0, 0, _p->_width, _p->_height);
        }
    }
    else
    {
        OSG_WARN << "Invalid width or height" << std::endl;
    }
}

/**
 * @brief get the width and height of the buffer
 * @return width new width,height new height
 */
void OsgCamtoGstbuf::getBuffersize(uint *width, uint *height)
{
    *width = _p->_width;
    *height = _p->_height;
}

/**
 * @brief set Gst buffer callback
 * @param object and callback function
 */
void OsgCamtoGstbuf::setBufferCallback(void *obj ,void (*cb)(void *,void *))
{
    gst_buffer_obj = obj;
    GstBuffer_callback = cb;
}

/**
 * @brief Get the camera associated and setup for the camtogst
 * @return instance of the camera
 */
osg::Camera *OsgCamtoGstbuf::getOrCreateCamera()
{
    if (!_p->_camera.valid())
    {
        _p->_camera = new osg::Camera;
        _p->_camera->setImplicitBufferAttachmentMask(0, 0);
        _p->_camera->setRenderOrder(osg::Camera::POST_RENDER);
        _p->_camera->setAllowEventFocus(false);

        const FboTraits &traits = getFramebufferTraits();
        GLbitfield clearMask = GL_COLOR_BUFFER_BIT;
        if (traits.depth > 0)
        {
            clearMask |= GL_DEPTH_BUFFER_BIT;
        }

        if (traits.stencil > 0)
        {
            clearMask |= GL_STENCIL_BUFFER_BIT;
        }

        _p->_camera->setClearMask(clearMask);
        _p->_camera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
        _p->_camera->setViewport(0, 0, _p->_width, _p->_height);
        _p->_camera->setName("H.264 capture");

        osg::ref_ptr<camtogstRTTPreDrawCallback> pdc = new camtogstRTTPreDrawCallback(this);
        _p->_camera->setPreDrawCallback(pdc.get());
        _p->qhandle = new camtogstRTTFinalDrawCallback(this, pdc.get());
        _p->_camera->setFinalDrawCallback(_p->qhandle);
    }

    return _p->_camera.get();
}

/**
 * @brief Attach the camtogst to an existing osg::Camera instance
 * @param camera
 *
 * After this call, getOrCreateCamera() will return the attached camera.
 * If a camera was previously attached to the camtogst, it will be detached
 * first. This can only be called if the camtogst is not running.
 */
void OsgCamtoGstbuf::attachToCamera(osg::Camera* camera)
{
    if (camera != 0)
    {
        if (_p->_camera.valid())
        {
            detachFromCamera(_p->_camera.get());
        }
        _p->_camera = camera;

        camera->setImplicitBufferAttachmentMask(0, 0);
        camera->setAllowEventFocus(false);
        camera->setName("H.264 capture");
        camera->setViewport(0, 0, _p->_width, _p->_height);

        osg::ref_ptr<camtogstRTTPreDrawCallback> pdc = new camtogstRTTPreDrawCallback(this, camera->getPreDrawCallback());
        camera->setPreDrawCallback(pdc.get());
        _p->qhandle = new camtogstRTTFinalDrawCallback(this, pdc.get(), camera->getFinalDrawCallback());
        camera->setFinalDrawCallback(_p->qhandle);
    }
    else
    {
        OSG_DEBUG << "Not attaching to NULL camera" << std::endl;
    }
}

/**
 * @brief Detach the camtogst from an existing osg::Camera instance
 * @param camera
 *
 * This can only be called if the camtogst is not running.
 */
void OsgCamtoGstbuf::detachFromCamera(osg::Camera* camera)
{
    if (camera != 0)
    {
        if (_p->_camera == camera)
        {
            camera->setAllowEventFocus(true);

            camtogstDrawCallback *edc = static_cast<camtogstDrawCallback*>(camera->getPreDrawCallback());
            camera->setPreDrawCallback(edc == 0 ? 0 : edc->getChainup());

            edc = static_cast<camtogstDrawCallback*>(camera->getFinalDrawCallback());
            camera->setFinalDrawCallback(edc == 0 ? 0 : edc->getChainup());
            _p->_camera = 0;
        }
        else
        {
            OSG_WARN << "Trying to detach an unknown camera form the camtogst" << std::endl;
        }
    }
    else
    {
        OSG_DEBUG << "Detaching from NULL camera, just removing it" << std::endl;
        _p->_camera = 0;
    }
}

void OsgCamtoGstbuf::setFramebufferTraits(const OsgCamtoGstbuf::FboTraits &traits)
{
    _p->_fboTraits = traits;
}

const OsgCamtoGstbuf::FboTraits &OsgCamtoGstbuf::getFramebufferTraits() const
{
    return _p->_fboTraits;
}

/**
 * @brief push a frame into the gstreamer pipeline
 * @param txo
 */
void OsgCamtoGstbuf::push(Texture_obj *txo)
{
    if (txo == 0)
    {
        return;
    }

    GstBuffer *gst_buffer = getGstBufferFromTex(txo->tex_id);

    if (gst_buffer != NULL)
    {
#if !GST_CHECK_VERSION(1,0,0)
        // To mark the Texture object that is wrapping the data pointers as
        // usable again, we attach a qdata to this buffer which will be destroyed
        // when the buffer is deleted.
        // in the PoolData destructor, the texture will be put back into the pool. It is
        // also responsible for deleting the gst_buffer_meta used above.
        GValue *value = g_new0(GValue, 1);
        g_value_init(value, pool_data_get_type());

        g_value_take_boxed(value, new PoolData(txo, this));


        GstStructure *structure = gst_structure_new(GST_BUFFER_QUARK, NULL);
        gst_structure_take_value(structure, "pool", value);
        gst_buffer_set_qdata(gst_buffer, g_quark_from_static_string(GST_BUFFER_QUARK), structure);
#else
        struct UserPtrData *data = NULL;
        data = g_slice_new0 (struct UserPtrData);
        data->osg_cam = (void *)this;
        data->tex = (void *)txo;

        gst_mini_object_set_qdata(GST_MINI_OBJECT(gst_buffer),
                                  g_quark_from_static_string(GST_BUFFER_QUARK),
                                  data,
                                  (GDestroyNotify)gst_buffer_final);
#endif
        if(gst_buffer_obj != NULL && GstBuffer_callback != NULL)
        {
            GstBuffer_callback(gst_buffer_obj, gst_buffer);
        }
        else
        {
            std::cout<<" callback function is NULL , please set callback "<<std::endl;
            gst_buffer_unref(gst_buffer);
        }
    }
    else
    {
        std::cout<<" gst_buffer is NULL "<<std::endl;
        txo->clearTextureObject(txo->context_id);
        releaseTex(txo->tex_id);
        txo->txo = NULL;
        txo->unref();
    }
}

void Texture_obj::clearTextureObject(uint context_id)
{
    _textureObjectBuffer[context_id].release();
}

OsgCamtoGstbuf::OsgCamtoGstbuf( const BufferTraits &txoTraits, const FboTraits &fboTraits)
    : generateTexGstbuf::GenTexGstbuf(txoTraits),
      _p(new OsgCamtoGstbufPrivate(this)),
      thread(0),
      texobj_queue()
{
    setFramebufferTraits(fboTraits);
}

OsgCamtoGstbuf::~OsgCamtoGstbuf()
{
    if(_p->_camera)
    {
        _p->_camera = NULL;
    }

    pthread_join(thread, NULL);
    while(!texobj_queue.empty())
    {
        Texture_obj *obj;
        obj = texobj_queue.front().obj;
        obj->clearTextureObject(obj->context_id);
        releaseTex(obj->tex_id);
        obj->txo = NULL;
        obj->unref();
        texobj_queue.pop();
    }

    pthread_mutex_destroy(&sync_lock);
    pthread_cond_destroy(&sync_cond);
}
} // namespace osgCamtoGstbuf
